/*
 * Copyright 2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include "fsl_common.h"
#include "core_zv2116.h"
#include "board.h"
#include "shared_memory_def.h"
#include "fsl_debug_console.h"
#include "rpmsg_lite.h"
#include "rpmsg_ns.h"

#define EZHV_NS_ANNOUNCE_STRING "MediaDomain - EZHV"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define SECURE_BOARD_LED_BLUE_GPIO     MSDK_REG_SECURE_ADDR(BOARD_LED_BLUE_GPIO)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

volatile int32_t g_has_received;
static cpu0_ezhv_parcel_t volatile cpu0_ezhv_parcel = {0};
static uint32_t g_remote_addr = 0;

/*******************************************************************************
 * Code
 ******************************************************************************/

/* Internal functions */
static int32_t rpmsg_ept_read_cb(void *payload, uint32_t payload_len, uint32_t src, void *priv)
{
    int32_t *has_received = priv;

    if (payload_len <= sizeof(cpu0_ezhv_parcel_t))
    {
        (void)memcpy((void *)&cpu0_ezhv_parcel, payload, payload_len);
        g_remote_addr = src;
        *has_received = 1;
    }
    return RL_RELEASE;
}

static void app_nameservice_isr_cb(uint32_t new_ept, const char *new_ept_name, uint32_t flags, void *user_data)
{
    uint32_t *data = (uint32_t *)user_data;
    if(data) {
        *data = new_ept;
    }
}

/*!
 * @brief Main function
 */
int main(void)
{
    struct rpmsg_lite_instance * p_ezhv_cpu0_rpmsg;
    struct rpmsg_lite_endpoint *p_ezhv_cpu0_ep = NULL;
    rpmsg_ns_handle cpu0_ns_handle;
	cpu0_ezhv_parcel_t ezhv_cpu0_parcel;
	uint32_t response_counter = 0;

    EzhvDbgConsole_Init(0, BOARD_DEBUG_UART_BAUDRATE, kSerialPort_Uart, BOARD_XTAL_SYS_CLK_HZ);
	PRINTF("[EZHV] Running\r\n");
    
    p_ezhv_cpu0_rpmsg = rpmsg_lite_remote_init((void *)CPU0_EZHV_RPMSG_LITE_BASE,
                                                        RL_PLATFORM_IMXRT700_M33_0_EZHV_LINK_ID, RL_NO_FLAGS);

    p_ezhv_cpu0_ep = rpmsg_lite_create_ept(p_ezhv_cpu0_rpmsg, EZHV_EPT_ADDR , rpmsg_ept_read_cb, (void *)&g_has_received);
    cpu0_ns_handle = rpmsg_ns_bind(p_ezhv_cpu0_rpmsg, app_nameservice_isr_cb, ((void *)0));
    (void)cpu0_ns_handle;
    rpmsg_lite_wait_for_link_up(p_ezhv_cpu0_rpmsg, RL_BLOCK);
    PRINTF("[EZHV] Link up!\r\n");

    SDK_DelayAtLeastUs(100000, SystemCoreClock); /* Wait CPU0 - rpmsg_lite_remote_init */ 
	rpmsg_ns_announce(p_ezhv_cpu0_rpmsg, p_ezhv_cpu0_ep, EZHV_NS_ANNOUNCE_STRING, (uint32_t)RL_NS_CREATE);
    SDK_DelayAtLeastUs(5000, SystemCoreClock); /* Wait CPU0 - rpmsg_lite_remote_init */ 
	PRINTF("[EZHV] Name service announce sent.\r\n");

    /* Define the init structure for the output LED pin*/
    gpio_pin_config_t led_config = {
        kGPIO_DigitalOutput,
        0,
    };

    /* Init LED Pin. */
    GPIO_PinInit(SECURE_BOARD_LED_BLUE_GPIO, BOARD_LED_BLUE_GPIO_PIN, &led_config);

    while (1)
    {
        g_has_received = 0;
        while(!g_has_received);
        PRINTF("[EZHV]  RECV %s\r\n", (char*)cpu0_ezhv_parcel.data);
        if(response_counter % 3 == 0)
        {
            GPIO_PortSet(SECURE_BOARD_LED_BLUE_GPIO, 1U << BOARD_LED_BLUE_GPIO_PIN);
        }
        else
        {
            GPIO_PortClear(SECURE_BOARD_LED_BLUE_GPIO, 1U << BOARD_LED_BLUE_GPIO_PIN);
        }
  
        snprintf((char *)ezhv_cpu0_parcel.data, 16, "EZ->C0:%d", response_counter++);
        
        rpmsg_lite_send(p_ezhv_cpu0_rpmsg, p_ezhv_cpu0_ep, g_remote_addr, 
			            (char *)ezhv_cpu0_parcel.data, sizeof(ezhv_cpu0_parcel), RL_BLOCK);
    }
}
