/*
 * Copyright (c) 2013 - 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017, 2025 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include "rpmsg_lite.h"
#include "rpmsg_queue.h"
#include "rpmsg_ns.h"
#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "dsp_support.h"
#include "core1_support.h"
#include "ezhv_support.h"
#include "fsl_ezhv.h"

#include "FreeRTOS.h"
#include "task.h"
#include "shared_memory_def.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define BOOT_SENSE_CM33 (1)
#define BOOT_HIFI4 (1)
#define BOOT_EZHV (1)
#define NON_SECURE_START (0x00100000)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void cpu0_task(void *param);
static void cpu1_task(void *param);
static void hifi4_task(void *param);
static void ezhv_task(void *param);
/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * @brief Main function
 */
int main(void)
{
    BOARD_InitAHBSC();
    /* Init board hardware. */
    BOARD_ConfigMPU();
    BOARD_InitPins();
    BOARD_BootClockRUN();
    BOARD_InitDebugConsole();
    CLOCK_EnableClock(kCLOCK_CompAccessRamArbiter1);
    CLOCK_EnableClock(kCLOCK_Gpio0);
    RESET_ClearPeripheralReset(kGPIO0_RST_SHIFT_RSTn);

    /* Enable GPIO access from DSP side */
    GPIO_EnablePinControlNonSecure(BOARD_LED_GREEN_GPIO, 1 << BOARD_LED_GREEN_GPIO_PIN);
    GPIO_EnablePinControlNonPrivilege(BOARD_LED_GREEN_GPIO, 1 << BOARD_LED_GREEN_GPIO_PIN);

    PRINTF("[CPU0] boot up - RPMSG \r\n");

#if BOOT_SENSE_CM33
    if (xTaskCreate(cpu1_task, "cpu1 task", configMINIMAL_STACK_SIZE + 256, NULL, 3, NULL) !=
        pdPASS)
    {
        PRINTF("\r\nFailed to create CPU1 task\r\n");
        while (1);
    }
#endif

#if BOOT_HIFI4
    if (xTaskCreate(hifi4_task, "hifi4 task", configMINIMAL_STACK_SIZE + 256, NULL, 4, NULL) !=
		pdPASS)
    {
        PRINTF("\r\nFailed to create HiFi4 task\r\n");
        while (1);
    }
#endif


#if BOOT_EZHV
    if (xTaskCreate(ezhv_task, "ezhv task", configMINIMAL_STACK_SIZE + 256, NULL, 4, NULL) !=
		pdPASS)
    {
        PRINTF("\r\nFailed to create EZHV task\r\n");
        while (1);
    }
#endif


    if (xTaskCreate(cpu0_task, "cpu0 task", configMINIMAL_STACK_SIZE, NULL, 2, NULL) !=
        pdPASS)
    {
        PRINTF("\r\nFailed to create CPU0 task\r\n");
        while (1);
    }
	
    vTaskStartScheduler();
}

static int32_t cpu0_ept_read_cb(void *payload, uint32_t payload_len, uint32_t src, void *priv)
{
    PRINTF("[CPU0]  RECV %s\r\n", payload);
    return 0;
}

static void app_nameservice_isr_cb(uint32_t new_ept, const char *new_ept_name, uint32_t flags, void *user_data)
{
    uint32_t *data = (uint32_t *)user_data;
    if(data) {
        *data = new_ept;
        PRINTF("Remote Name, %d - %s\r\n", new_ept, new_ept_name);
    }
}

static void ezhv_task(void *param)
{
    volatile uint32_t ezhv_address = 0U;
    struct rpmsg_lite_instance *p_cpu0_ezhv_rpmsg = NULL;
    struct rpmsg_lite_endpoint *p_cpu0_ezhv_ep = NULL;
    rpmsg_ns_handle ezhv_ns_handle;
    cpu0_ezhv_parcel_t cpu0_ezhv_parcel;
    uint32_t message_counter = 0;
	
    vTaskDelay(100 / portTICK_PERIOD_MS);
    BOARD_EZHV_SetSecurePriv();
    BOARD_EZHV_Init();
    vTaskDelay(30 / portTICK_PERIOD_MS);
    p_cpu0_ezhv_rpmsg = rpmsg_lite_master_init((void *)CPU0_EZHV_RPMSG_LITE_BASE, 
		                        CPU0_EZHV_SH_MEM_TOTAL_SIZE, RL_PLATFORM_IMXRT700_M33_0_EZHV_LINK_ID, RL_NO_FLAGS);
    
    p_cpu0_ezhv_ep = rpmsg_lite_create_ept(p_cpu0_ezhv_rpmsg, CPU0_EPT_ADDR, cpu0_ept_read_cb, NULL);

    ezhv_ns_handle = rpmsg_ns_bind(p_cpu0_ezhv_rpmsg, app_nameservice_isr_cb, (void *)&ezhv_address);
    (void)ezhv_ns_handle;

    while(ezhv_address != EZHV_EPT_ADDR)
    {
        vTaskDelay(5);
    }

    while(1)
    {		
        snprintf((char *)cpu0_ezhv_parcel.data, 16, "C0->EZ:%d", message_counter++);
        rpmsg_lite_send(p_cpu0_ezhv_rpmsg, p_cpu0_ezhv_ep, EZHV_EPT_ADDR,
			            (char *)&cpu0_ezhv_parcel, sizeof(cpu0_ezhv_parcel), RL_BLOCK);
        vTaskDelay(1000);
    }
}

static void cpu0_task(void *param)
{
    while(1)
    {		
        vTaskDelay(200);
    }
}

#if BOOT_SENSE_CM33
static void cpu1_task(void *param)
{
    volatile uint32_t cpu1_address = 0U;
    struct rpmsg_lite_instance *p_cpu0_cpu1_rpmsg = NULL;
    struct rpmsg_lite_endpoint *p_cpu0_cpu1_ep = NULL;
    rpmsg_ns_handle cpu1_ns_handle;
    cpu0_cpu1_parcel_t cpu0_cpu1_parcel;
    uint32_t message_counter = 0;

    BOARD_CopyCore1Image(CORE1_BOOT_ADDRESS);
    BOARD_ReleaseCore1Power();
    BOARD_BootCore1((CORE1_BOOT_ADDRESS & 0x0FFFFFFF), (CORE1_BOOT_ADDRESS & 0x0FFFFFFF));

    SDK_DelayAtLeastUs(30000, SystemCoreClock); /* Wait CPU1 - rpmsg_lite_remote_init */ 
    p_cpu0_cpu1_rpmsg = rpmsg_lite_master_init((void *)CPU0_CPU1_RPMSG_LITE_BASE, 
		                        CPU0_CPU1_SH_MEM_TOTAL_SIZE, RL_PLATFORM_IMXRT700_M33_0_M33_1_LINK_ID, RL_NO_FLAGS);
    
    p_cpu0_cpu1_ep = rpmsg_lite_create_ept(p_cpu0_cpu1_rpmsg, CPU0_EPT_ADDR, cpu0_ept_read_cb, NULL);

    cpu1_ns_handle = rpmsg_ns_bind(p_cpu0_cpu1_rpmsg, app_nameservice_isr_cb, (void *)&cpu1_address);
    (void)cpu1_ns_handle;

    while(cpu1_address != CPU1_EPT_ADDR)
    {
        vTaskDelay(5);
    }

    while(1)
    {		
        snprintf((char *)cpu0_cpu1_parcel.data, 16, "C0->C1:%d", message_counter++);
        rpmsg_lite_send(p_cpu0_cpu1_rpmsg, p_cpu0_cpu1_ep, CPU1_EPT_ADDR,
			            (char *)&cpu0_cpu1_parcel, sizeof(cpu0_cpu1_parcel), RL_BLOCK);
        vTaskDelay(1000);
    }
}
#endif

#if BOOT_HIFI4
static void hifi4_task(void *param)
{
    volatile uint32_t hifi4_address = 0U;
    struct rpmsg_lite_instance *p_cpu0_hifi4_rpmsg = NULL;
    struct rpmsg_lite_endpoint *p_cpu0_hifi4_ep = NULL;
    rpmsg_ns_handle hifi4_ns_handle;
    cpu0_hifi4_parcel_t cpu0_hifi4_parcel;
    uint32_t message_counter = 0;

    BOARD_DSP_Init();
    SDK_DelayAtLeastUs(10000, SystemCoreClock); /* Wait HiFi4 - rpmsg_lite_remote_init */     

    p_cpu0_hifi4_rpmsg = rpmsg_lite_master_init((void *)CPU0_HIFI4_RPMSG_LITE_BASE, 
		                        CPU0_HIFI4_SH_MEM_TOTAL_SIZE, RL_PLATFORM_IMXRT700_M33_0_HIFI4_LINK_ID, RL_NO_FLAGS);
    
    p_cpu0_hifi4_ep = rpmsg_lite_create_ept(p_cpu0_hifi4_rpmsg, CPU0_EPT_ADDR, cpu0_ept_read_cb, NULL);

    hifi4_ns_handle = rpmsg_ns_bind(p_cpu0_hifi4_rpmsg, app_nameservice_isr_cb, (void *)&hifi4_address);
    (void)hifi4_ns_handle;

    while(hifi4_address != HIFI4_EPT_ADDR)
    {
        vTaskDelay(5);
    }

    while(1)
    {
        snprintf((char *)cpu0_hifi4_parcel.data, 16, "C0->H4:%d", message_counter++);
        rpmsg_lite_send(p_cpu0_hifi4_rpmsg, p_cpu0_hifi4_ep, HIFI4_EPT_ADDR,
			           (char *)&cpu0_hifi4_parcel, sizeof(cpu0_hifi4_parcel), RL_BLOCK);
        vTaskDelay(1000);
    }
}
#endif