/*
 * Copyright 2018-2019, 2025 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <xtensa/config/core.h>
#include <xtensa/xos.h>
#include <stdio.h>

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "board.h"
#include "shared_memory_def.h"
#include "fsl_inputmux.h"

#include "rpmsg_lite.h"
#include "rpmsg_queue.h"
#include "rpmsg_ns.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define HIFI1_NS_ANNOUNCE_STRING "Sense - HiFi1"

#define DSP_THREAD_PRIORITY (1)
#define DSP_THREAD_STACK_SIZE (10 * 1024)
static uint8_t dsp_thread_stack[DSP_THREAD_STACK_SIZE];
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
int DSP_Main(void *arg, int wake_value);

static void XOS_Init(void)
{
    xos_set_clock_freq(XOS_CLOCK_FREQ);
    xos_start_system_timer(-1, 0);
}
/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{
    XosThread thread_main;
    /* Init board hardware. */
    CLOCK_SetXtalFreq(BOARD_XTAL_SYS_CLK_HZ);
    CLOCK_EnableClock(kCLOCK_InputMux);
    INPUTMUX_Init(INPUTMUX1);
    INPUTMUX_AttachSignal(INPUTMUX1, 1U, kINPUTMUX_Mu3BToDspInterrupt); /* MU3B for CPU1/HiFi1 */
    INPUTMUX_Deinit(INPUTMUX1);
    XOS_Init();
    DbgConsole_Init_Alt(BOARD_DEBUG_UART_INSTANCE, kSerialPort_Uart);

    LED_RED_INIT(LOGIC_LED_OFF);
    PRINTF("[HiFi1] Running\r\n");

    xos_thread_create(&thread_main, NULL, DSP_Main, NULL, "DSP Main", dsp_thread_stack, DSP_THREAD_STACK_SIZE,
                      DSP_THREAD_PRIORITY, 0, 0);

    /* Start XOS scheduler - does not return */
    xos_start(0);
}

static void app_nameservice_isr_cb(uint32_t new_ept, const char *new_ept_name, uint32_t flags, void *user_data)
{
    uint32_t *data = (uint32_t *)user_data;
    if(data) {
        *data = new_ept;
    }
}

int DSP_Main(void *arg, int wake_value)
{
    struct rpmsg_lite_instance * p_hifi1_cpu1_rpmsg;
    struct rpmsg_lite_endpoint *p_hifi1_cpu1_ep = NULL;
    rpmsg_queue_handle hifi1_cpu1_queue;
    rpmsg_ns_handle cpu1_ns_handle;
	cpu1_hifi1_parcel_t cpu1_hifi1_parcel;
    hifi1_cpu1_parcel_t hifi1_cpu1_parcel;
    uint32_t response_counter = 0;
    int32_t status;

	PRINTF("[HiFi1] - DSP Main\r\n");

    p_hifi1_cpu1_rpmsg = rpmsg_lite_remote_init((void *)CPU1_HIFI1_RPMSG_LITE_BASE,
                                                        RL_PLATFORM_IMXRT700_M33_1_HIFI1_LINK_ID, RL_NO_FLAGS);

    hifi1_cpu1_queue = rpmsg_queue_create(p_hifi1_cpu1_rpmsg);
    p_hifi1_cpu1_ep = rpmsg_lite_create_ept(p_hifi1_cpu1_rpmsg, HIFI1_EPT_ADDR, rpmsg_queue_rx_cb, hifi1_cpu1_queue);
    cpu1_ns_handle = rpmsg_ns_bind(p_hifi1_cpu1_rpmsg, app_nameservice_isr_cb, ((void *)0));
    (void)cpu1_ns_handle;
    rpmsg_lite_wait_for_link_up(p_hifi1_cpu1_rpmsg, RL_BLOCK);
    PRINTF("[HiFi1] Link up!\r\n");

    xos_thread_sleep_msec(100);
	rpmsg_ns_announce(p_hifi1_cpu1_rpmsg, p_hifi1_cpu1_ep, HIFI1_NS_ANNOUNCE_STRING, (uint32_t)RL_NS_CREATE);
    xos_thread_sleep_msec(10);
	PRINTF("[HiFi1] Name service announce sent.\r\n");

    while (1)
    {
        status = rpmsg_queue_recv(p_hifi1_cpu1_rpmsg, hifi1_cpu1_queue, NULL, (char *)&cpu1_hifi1_parcel, 
                                  sizeof(cpu1_hifi1_parcel), NULL, RL_BLOCK);
     	if (status != RL_SUCCESS)
     	{
     		PRINTF("Failed to get item from RPMsg queue.\r\n");
     	} else {
            PRINTF("[HiFi1] RECV %s\r\n", (char*)cpu1_hifi1_parcel.data);
        }

        if(response_counter % 3 == 2)
        {
            LED_RED_ON();
        }
        else
        {
            LED_RED_OFF();
        }
        snprintf((char *)hifi1_cpu1_parcel.data, 16, "H1->C1:%d", response_counter++);
        
        rpmsg_lite_send(p_hifi1_cpu1_rpmsg, p_hifi1_cpu1_ep, CPU1_EPT_ADDR,
			            (char *)hifi1_cpu1_parcel.data, sizeof(hifi1_cpu1_parcel), RL_BLOCK);
    }
}
